// client/composables/voice-sfu/index.js
import {io} from 'socket.io-client'
import {ref, onUnmounted} from 'vue'
import * as mediasoupClient from 'mediasoup-client'
import {useNuxtApp} from 'nuxt/app'

import {createAudioModule} from './audio'
import {createMeterModule} from './meter'
import {createRpc} from './rpc'
import {createMediaSoup} from './mediasoup'
import {Env} from './env'
import {createSession} from './session'
import {useRoomStore} from "~/stores/room";

/**
 * useVoiceRoom1 – API بیرونی بدون تغییر:
 *  joinRoom, leaveRoom, role, users, localStream,
 *  requestAudioUnlock, audioUnlocked,
 *  raiseHand, sendAccessMic, clearRoom, muteListener,
 *  sendReaction, setUserRoleName, setUserSeat, changeMicStatus,
 *  isIOS, isAndroid, speakingMap, volumeMap
 */
export const useVoiceRoom1 = (roomId, currentUserId, roleItem, campaignUsers) => {
    console.log('[SFU] init composable', {roomId, currentUserId, roleItem})

    // ----- State (بدون تغییر)
    const role = ref(roleItem === 'host' ? 'speaker' : 'listener')
    const users = ref({})
    const myId = ref(null)
    const localStream = ref(null)

    const speakingMap = ref({})
    const volumeMap = ref({})
    const audioUnlocked = ref(false)

    const {$sweetalert,$api} = useNuxtApp()

    // MediaSoup holders (by reference)
    const holders = {
        device: null,
        sendTransport: null,
        recvTransport: null,
        currentSender: null
    }

    const consumers = new Map()
    const consumerAudios = new Map()

    // Socket
    const socket = io('https://carjoo.com', {transports: ['websocket'], secure: true})

    // ---- Modules
    const AudioModule = createAudioModule({audioUnlockedRef: audioUnlocked})
    const MeterModule = createMeterModule({
        AudioModule,
        speakingMapRef: speakingMap,
        volumeMapRef: volumeMap,
        myIdRef: myId,
        socket,     // 👈 اضافه شد
        roomId
})
    const Rpc = createRpc(socket)
    const MediaSoup = createMediaSoup({
        roomId,
        refs: {role, myId, localStream, audioUnlocked},
        maps: {consumers, consumerAudios},
        holders,
        modules: {Rpc, MeterModule, AudioModule},
        mediasoupClient
    })

    const Session = createSession({
        socket,
        roomId,
        roleItem,
        refs: {role, users, myId, localStream},
        maps: {consumers, consumerAudios},
        holders,
        modules: {MediaSoup, MeterModule, Env},
        ui: {$sweetalert,$api}
    })

    socket.on('speaking-update', ({ userId, speaking, volume }) => {
        // 1) نقشه‌های جدا برای UI‌های ساده (حلقه‌ی سبز، نوار حجم و ...)
        speakingMap.value = { ...speakingMap.value, [userId]: !!speaking }
        volumeMap.value   = { ...volumeMap.value, [userId]: typeof volume === 'number' ? volume : 0 }

        // 2) اگر UI از users[] می‌خواند، آن را هم به‌روز کن تا رندر دوباره انجام شود
        if (users.value?.[userId]) {
            users.value = {
                ...users.value,
                [userId]: {
                    ...users.value[userId],
                    speaking: !!speaking,
                    volume: typeof volume === 'number' ? volume : 0
                }
            }
        }
    })

    // --- Parity Actions (بدون تغییر رفتار)
    const raiseHand = () => {
        console.log('[SFU] raise-hand')
        socket.emit('raise-hand', roomId)
    }

    const sendAccessMic = (user_id) => {
        console.log('[SFU] access-mic', {user_id})
        socket.emit('access-mic', {room: roomId, userId: user_id})
    }

    const callRequest = (user_id) => {
        console.log('[SFU] call-request', {user_id})
        socket.emit('call-request', {room: roomId, userId: user_id})
    }

    const clearRoom = () => {
        console.log('[SFU] clear-room emitted')
        socket.emit('clear-room', roomId)

        for (const [uid, a] of consumerAudios) {
            a.pause()
            a.srcObject = null
            a.remove()
            console.log('[SFU] (clearRoom) audio removed', {uid})
        }
    }

    const muteListener = (user_id = '') => {
        if (user_id === '') {
            console.log('[SFU] force-listeners')
            socket.emit('force-listeners', roomId)
        } else {
            console.log('[SFU] force-listener', {user_id})
            socket.emit('force-listener', {roomId, userId: user_id})
        }
    }

    const sendReaction = (emoji) => {
        console.log('[SFU] send-reaction', {emoji})
        socket.emit('send-reaction', {room: roomId, emoji})
    }

    const setUserRoleName = (targetUserId, newRoleName) => {
        console.log('[SFU] set-role-name', {targetUserId, newRoleName})
        socket.emit('set-role-name', {roomId, userId: targetUserId, roleName: newRoleName})
    }

    const setUserSeat = (targetUserId, seat) => {
        console.log('[SFU] set-seat', {targetUserId, seat})
        socket.emit('set-seat', {roomId, userId: targetUserId, seat})
    }

    // Mic toggle parity (نرم: track.enabled)
    const changeMicStatus = async (micEnabled) => {
        console.log('[SFU] changeMicStatus called', {micEnabled})
        const t = localStream.value?.getAudioTracks?.()[0]
        const newState = !micEnabled
        if (t) {
            t.enabled = newState
            console.log('[SFU] track.enabled set to', newState)
        }
        socket.emit('mic-change-state', {room: roomId, micState: newState})
    }

    // iOS/Android parity
    const isIOS = Env.isIOS
    const isAndroid = Env.isAndroid

    // --- joinRoom با همان داده‌های قبلی (image/name از campaignUsers)
    const joinRoom = () => {
        console.log('[SFU] joinRoom called; role=', role.value)
        AudioModule.bindGestureUnlock(document)
        Session.setupSocket()
        socket.emit('join', {
            room: roomId,
            role: role.value,
            userId: currentUserId,
            roleName: roleItem,
            device: Env.detectDevice(),
            micState: true,
            image: campaignUsers.value.find((i) => i.id === currentUserId)?.image,
            skin: campaignUsers.value.find((i) => i.id === currentUserId)?.skin,
            ghab: campaignUsers.value.find((i) => i.id === currentUserId)?.ghab,
            name: campaignUsers.value.find((i) => i.id === currentUserId)?.name || 'بدون‌نام'
        })
        console.log('[SFU] join emitted')
    }

    const leaveRoom = (user_id = '') => {
        console.log('[SFU] leaveRoom called', {user_id, asHost: roleItem === 'host'})
        if (user_id && roleItem === 'host') {
            socket.emit('kick-user', {room: roomId, targetUserId: user_id})
            console.log('[SFU] kick-user emitted', {targetUserId: user_id})
            return
        }
        socket.emit('leave-room', roomId)
        console.log('[SFU] leave-room emitted')
        Session.teardown()
    }

    onUnmounted(() => {
        console.log('[SFU] onUnmounted -> teardown')
        Session.teardown()
    })

    // ===== API خروجی دقیقاً مثل قبل =====
    return {
        // core
        joinRoom,
        leaveRoom,
        role,
        users,
        localStream,

        // audio unlock
        requestAudioUnlock: AudioModule.requestAudioUnlock,
        bindGestureUnlock: AudioModule.bindGestureUnlock,
        audioUnlocked,

        // parity actions
        raiseHand,
        sendAccessMic,
        callRequest,
        clearRoom,
        muteListener,
        sendReaction,
        setUserRoleName,
        setUserSeat,
        changeMicStatus,
        isIOS,
        isAndroid,

        // meters (اختیاری برای UI)
        speakingMap,
        volumeMap
    }
}
