// client/composables/voice-sfu/session.js
import { useRoomStore } from '~/stores/room'

export function createSession({
                                  socket,
                                  roomId,
                                  roleItem,
                                  refs,   // { role, users, myId, localStream }
                                  maps,   // { consumers, consumerAudios }
                                  holders, // { sendTransport, recvTransport, currentSender }
                                  modules, // { MediaSoup, MeterModule, Env }
                                  ui,     // { $sweetalert }
                                  logger = console
                              }) {
    const {role, users, myId, localStream} = refs
    const {consumers, consumerAudios} = maps
    const {MediaSoup, MeterModule, Env} = modules
    let {sendTransport, recvTransport, currentSender} = holders
    const {$sweetalert,$api} = ui

    const setupSocket = () => {
        const roomStore = useRoomStore()


        logger.log('[SFU] setupSocket')

        socket.on('connect', () => logger.log('[SFU] socket connected', {id: socket.id}))
        socket.on('disconnect', (reason) => logger.log('[SFU] socket disconnected', {reason}))

        socket.on('joined', async ({id, role: assignedRole, users: joinedUsers}) => {
            logger.log('[SFU] joined event', {id, assignedRole, usersCount: Object.keys(joinedUsers || {}).length})
            myId.value = id
            role.value = assignedRole
            users.value = joinedUsers

            if (role.value === 'speaker') {
                await MediaSoup.startSpeaker()
            }
            const speakers = Object.entries(joinedUsers)
                .filter(([_, u]) => u.role === 'speaker').map(([uid]) => uid)
            logger.log('[SFU] existing speakers', speakers)
            for (const uid of speakers) {
                if (uid !== myId.value) await MediaSoup.consumeFrom(uid)
            }
        })
        socket.on('new-producer', async (payload) => {
            // 👇 هر اسمی که سرور فرستاد رو پشتیبانی کن
            const producerUserId = payload?.userId ?? payload?.producerUserId ?? payload?.from ?? payload?.id ?? payload?.uid ?? null

            logger.log('[SFU] new-producer', {
                producerUserId, myId: myId.value, myRole: role.value, raw: payload
            })
            if (!producerUserId) {
                logger.warn('[SFU] new-producer without user id', payload)
                return
            }
            if (String(producerUserId) === String(myId.value)) return
            // ✅ حتی در نقش speaker هم صدای دیگران را بگیر
            await MediaSoup.consumeFrom(producerUserId)
        })

        socket.on('producer-closed', ({userId: producerUserId}) => {
            logger.log('[SFU] producer-closed', {producerUserId})
            const audio = consumerAudios.get(producerUserId)
            if (audio) {
                audio.pause()
                audio.srcObject = null
                audio.remove()
                consumerAudios.delete(producerUserId)
            }
            const c = consumers.get(producerUserId)
            if (c) {
                try {
                    c.close()
                } catch {
                }
                consumers.delete(producerUserId)
            }
        })

        socket.on('users-update', (u) => {
            logger.log('[SFU] users-update', {count: Object.keys(u || {}).length})
            users.value = u
        })

        socket.on('role-changed', async (newRole) => {
            logger.log('[SFU] role-changed', {from: role.value, to: newRole})
            role.value = newRole
            if (newRole === 'speaker') {
                await MediaSoup.startSpeaker()
            } else {
                MeterModule.stopLocalMeter()
                if (localStream.value) {
                    logger.log('[SFU] stopping local tracks')
                    localStream.value.getTracks().forEach((t) => t.stop())
                    localStream.value = null
                }
                if (holders.sendTransport) {
                    logger.log('[SFU] closing sendTransport')
                    try {
                        await holders.sendTransport.close()
                    } catch {
                    }
                    holders.sendTransport = null
                    holders.currentSender = null
                    sendTransport = null
                    currentSender = null
                }
            }
        })

        socket.on('hand-raised', ({from}) => {
            const requester = users.value[from]
            if (!requester) return
            $sweetalert.confirm(() => {
                socket.emit('promote-to-speaker', {room: roomId, targetId: from})
            }, `کاربر ${requester.name} درخواست صحبت دارد. قبول می‌کنید؟`)
        })

        socket.on('reaction', ({ userId, emoji }) => {
            const el = document.querySelector(`#reaction-${userId}`)
            if (el) {
                el.innerText = emoji
                el.classList.add('show')
                setTimeout(() => {
                    el.classList.remove('show')
                    el.innerText = ''
                }, 3000)
            } else {
                console.warn('❌ Reaction element not found for userId:', userId)
            }
        })

        socket.on('access-mic-sent', ({from}) => {
            const requester = users.value[from]
            if (!requester) return
            $sweetalert.confirm(() => {
                socket.emit('promote-to-speaker', {room: roomId, targetId: myId.value})
            }, 'به شما اجازه صحبت داده شد. ارتقا به Speaker؟')
        })

        socket.on('call-requested', async ({from}) => {
            const requester = users.value[from]
            if (!requester) return
            await $sweetalert.confirm(async () => {
                let slug = '';

                await $api.post('/user/room/create',{title: 'تماس اینترنتی', type: 'call',category_id: 1,user_id: from}).then((value) => {
                    if(value.data.status) {
                        slug = value.data.data?.slug;
                    }
                });

                socket.emit('go-call', {room: roomId, userId: from,slug: slug})
                window.location.href = "https://ios.artjoo.com/club/call/" + slug;
            }, `کاربر ${requester.name} درخواست تماس خصوصی دارد. قبول می‌کنید؟`)
        })

        socket.on('go-call-requested', async ({slug}) => {
            window.location.href = "https://ios.artjoo.com/club/call/" + slug;
        })

        socket.on('room-cleared', () => {
            logger.log('[SFU] room-cleared')
            try {
                roomStore.reset()
                logger.log('[SFU] roomStore.reset called')
            } catch (e) {
                logger.warn('[SFU] roomStore.reset failed', e)
            }
            teardown()
        })

        socket.on('error', (e) => logger.error('[SFU] socket error', e))
    }

    const joinRoom = () => {
        logger.log('[SFU] joinRoom called; role=', role.value)
        setupSocket()
        socket.emit('join', {
            room: roomId,
            role: role.value,
            userId: myId.value ? myId.value : undefined, // اختیاری: اما رفتار بیرونی تغییری نمی‌کند
            roleName: roleItem,
            device: Env.detectDevice(),
            micState: true,
            // image/name از کد فراخوان اصلی تزریق می‌شود (بیرون)
        })
        logger.log('[SFU] join emitted')
    }

    const teardown = () => {
        logger.log('[SFU] teardown start')

        for (const [uid, c] of consumers) {
            try {
                c.close()
            } catch {
            }
            logger.log('[SFU] consumer closed', {uid})
        }
        consumers.clear()

        for (const [uid, a] of consumerAudios) {
            a.pause()
            a.srcObject = null
            a.remove()
            logger.log('[SFU] audio elem removed', {uid})
        }
        consumerAudios.clear()

        try {
            holders.sendTransport?.close();
            if (holders.sendTransport) logger.log('[SFU] sendTransport closed')
        } catch {
        }
        try {
            holders.recvTransport?.close();
            if (holders.recvTransport) logger.log('[SFU] recvTransport closed')
        } catch {
        }
        holders.sendTransport = null
        holders.recvTransport = null
        holders.currentSender = null
        sendTransport = null
        recvTransport = null
        currentSender = null

        MeterModule.stopLocalMeter()
        if (localStream.value) {
            logger.log('[SFU] stop localStream tracks')
            localStream.value.getTracks().forEach((t) => t.stop())
            localStream.value = null
        }

        socket.off('joined')
        socket.off('new-producer')
        socket.off('producer-closed')
        socket.off('users-update')
        socket.off('role-changed')
        socket.off('room-cleared')
        logger.log('[SFU] socket listeners removed')

        socket.disconnect()
        logger.log('[SFU] socket disconnected (manual)')
    }

    return {setupSocket, joinRoom, teardown}
}
