// client/composables/voice-sfu/mediasoup.js

export function createMediaSoup({
                                    roomId,
                                    refs,          // { role, myId, localStream, audioUnlocked }
                                    maps,          // { consumers, consumerAudios }
                                    holders,       // { device, sendTransport, recvTransport, currentSender }  (by reference)
                                    modules,       // { Rpc, MeterModule, AudioModule }
                                    mediasoupClient,
                                    logger = console
                                }) {
    const {role, myId, localStream, audioUnlocked} = refs
    const {consumers, consumerAudios} = maps
    let {device, sendTransport, recvTransport, currentSender} = holders
    const {Rpc, MeterModule, AudioModule} = modules

    const ensureDevice = async () => {
        logger.log('[SFU] ensureDevice start; hasDevice=', !!device)
        if (device) return device
        const {ok, rtpCapabilities, error} = await Rpc.askServer('get-rtp-capabilities', {roomId})
        if (!ok) {
            logger.error('[SFU] get-rtp-capabilities failed', error)
            throw new Error(error || 'get-rtp-capabilities failed')
        }
        const d = new mediasoupClient.Device()
        await d.load({routerRtpCapabilities: rtpCapabilities})
        device = d
        holders.device = device
        logger.log('[SFU] device loaded', {rtpCaps: !!rtpCapabilities})
        return device
    }

    const createTransport = async (direction) => {
        logger.log('[SFU] createTransport start', {direction})
        const {ok, id, iceParameters, iceCandidates, dtlsParameters, error} =
            await Rpc.askServer('create-transport', {roomId, direction})
        if (!ok) {
            logger.error('[SFU] create-transport failed', error)
            throw new Error(error)
        }

        const transport =
            direction === 'send'
                ? device.createSendTransport({id, iceParameters, iceCandidates, dtlsParameters})
                : device.createRecvTransport({id, iceParameters, iceCandidates, dtlsParameters})

        logger.log('[SFU] transport created', {direction, id: transport.id})

        transport.on('connect', async ({dtlsParameters}, callback, errback) => {
            logger.log('[SFU] transport.on(connect)', {direction, transportId: transport.id})
            const ans = await Rpc.askServer('connect-transport', {transportId: transport.id, dtlsParameters})
            if (ans.ok) {
                logger.log('[SFU] transport connected', {direction, transportId: transport.id})
                callback()
            } else {
                logger.error('[SFU] transport connect error', ans.error)
                errback(new Error(ans.error))
            }
        })

        if (direction === 'send') {
            transport.on('produce', async ({kind, rtpParameters}, callback, errback) => {
                logger.log('[SFU] transport.on(produce) ->', {kind, transportId: transport.id})
                const ans = await Rpc.askServer('produce', {
                    roomId,
                    transportId: transport.id,
                    kind,
                    rtpParameters,
                    userId: myId.value
                })
                if (ans.ok) {
                    logger.log('[SFU] produced OK', {kind, id: ans.id})
                    callback({id: ans.id})
                } else {
                    logger.error('[SFU] produce error', ans.error)
                    errback(new Error(ans.error))
                }
            })
        }

        transport.on('connectionstatechange', (state) => {
            logger.log('[SFU] transport state', {direction, state})
        })

        return transport
    }

    // --- Speaker flow
    const startSpeaker = async () => {
        logger.log('[SFU] startSpeaker called; role=', role.value, 'unlocked=', audioUnlocked.value)
        if (role.value !== 'speaker') return
        if (!audioUnlocked.value) {
            logger.warn('[SFU] not unlocked yet; trying to unlock')
            try {
                await AudioModule.requestAudioUnlock()
            } catch {
            }
        }

        await ensureDevice()
        try {
            localStream.value = await navigator.mediaDevices.getUserMedia({
                audio: {echoCancellation: true, noiseSuppression: true, autoGainControl: true}
            })
            logger.log('[SFU] gotUserMedia OK', {tracks: localStream.value?.getAudioTracks()?.length})
        } catch (e) {
            logger.error('[SFU] getUserMedia failed', e)
            return
        }

        sendTransport = await createTransport('send')
        holders.sendTransport = sendTransport
        logger.log('[SFU] sendTransport ready', {id: sendTransport.id})

        const track = localStream.value.getAudioTracks()[0]
        logger.log('[SFU] producing audio track...', {enabled: track?.enabled, readyState: track?.readyState})
        const producer = await sendTransport.produce({track})
        logger.log('[SFU] producer created', {id: producer?.id})

        currentSender = producer?.transport?._connection?._sender || null
        holders.currentSender = currentSender
        if (currentSender) logger.log('[SFU] currentSender captured')

        const a = document.createElement('audio')
        a.srcObject = localStream.value
        a.muted = true
        a.autoplay = true
        a.playsInline = true
        a.classList.add('hidden-audio', 'speaker')
        a.style.opacity = '0'
        document.body.appendChild(a)
        await AudioModule.safePlay(a)
        logger.log('[SFU] local monitor audio appended')

        MeterModule.startLocalMeter(localStream.value)
    }

    // --- Listener flow
    const ensureRecvTransport = async () => {
        logger.log('[SFU] ensureRecvTransport; hasRecv=', !!recvTransport)
        if (recvTransport) return recvTransport
        await ensureDevice()
        recvTransport = await createTransport('recv')
        holders.recvTransport = recvTransport
        logger.log('[SFU] recvTransport ready', {id: recvTransport.id})
        return recvTransport
    }

    const consumeFrom = async (producerUserId,retryCount = 0) => {
        logger.log('[SFU] consumeFrom called', {producerUserId})
        try {
            if (!audioUnlocked.value) {
                logger.warn('[SFU] Audio not unlocked; trying to unlock')
                try {
                    await AudioModule.requestAudioUnlock()
                } catch {
                }
            }
            await ensureRecvTransport()

            const ans = await Rpc.askServer('consume', {
                roomId,
                transportId: recvTransport.id,
                rtpCapabilities: device.rtpCapabilities,
                producerUserId
            })

            const me = String(myId?.value ?? myId)   // هرطور دسترسی داری
            if (String(producerUserId) === me) {
                console.log('[SFU] skip self-consume for', producerUserId)
                return
            }
            if (!ans.ok) {
                logger.warn('[SFU] consume failed:', ans.error)
                // ⏳ 300ms بعد دوباره تلاش کن (حداکثر 3 بار)
                if (ans.error === 'producer not found' && (retryCount ?? 0) < 3) {
                    setTimeout(() => consumeFrom(producerUserId, (retryCount ?? 0) + 1), 300)
                }
                return
            }

            const consumer = await recvTransport.consume({
                id: ans.id,
                producerId: ans.producerId,
                kind: ans.kind,
                rtpParameters: ans.rtpParameters
            })
            logger.log('[SFU] consumer created', {id: consumer.id, producerId: ans.producerId, kind: ans.kind})
            consumers.set(producerUserId, consumer)

            const stream = new MediaStream([consumer.track])
            consumer.track.onmute = () => logger.log('[SFU] consumer track muted', {producerUserId})
            consumer.track.onunmute = () => logger.log('[SFU] consumer track UNMUTED', {producerUserId})
            consumer.on('producerpauconnectse', () => logger.log('[SFU] producer paused', {producerUserId}))
            consumer.on('producerresume', () => logger.log('[SFU] producer resumed', {producerUserId}))

            let audio = consumerAudios.get(producerUserId)
            if (!audio) {
                audio = document.createElement('audio')
                audio.autoplay = true
                audio.playsInline = true
                audio.controls = true
                audio.classList.add('hidden-audio', 'listener')
                audio.style.opacity = '0'
                document.body.appendChild(audio)
                consumerAudios.set(producerUserId, audio)
                logger.log('[SFU] listener audio element created', {producerUserId})
            }
            audio.srcObject = stream
            audio.muted = false
            audio.volume = 1
            await AudioModule.safePlay(audio)

            if ('setSinkId' in HTMLMediaElement.prototype) {
                try {
                    await audio.setSinkId('default')
                    logger.log('[SFU] setSinkId default OK')
                } catch (e) {
                    logger.warn('[SFU] setSinkId error', e)
                }
            }

            try {
                await consumer.resume()
                logger.log('[SFU] consumer resume OK')
            } catch (e) {
                logger.warn('[SFU] consumer resume error', e)
            }
            const r = await Rpc.askServer('resume-consumer', {roomId, producerUserId })
            if (!r.ok) {
                logger.warn('[SFU] resume-consumer failed', r.error)
            } else {
                logger.log('[SFU] resume-consumer OK', {consumerId: consumer.id})
            }

            consumer.on('transportclose', () => {
                logger.log('[SFU] consumer transportclose', {producerUserId})
                audio.pause()
                audio.srcObject = null
                audio.remove()
                consumerAudios.delete(producerUserId)
                consumers.delete(producerUserId)
            })
            consumer.on('producerclose', () => {
                logger.log('[SFU] consumer producerclose', {producerUserId})
                audio.pause()
                audio.srcObject = null
                audio.remove()
                try {
                    consumer.close()
                } catch {
                }
                consumerAudios.delete(producerUserId)
                consumers.delete(producerUserId)
            })
        } catch (e) {
            logger.warn('[SFU] consumeFrom error', e)
        }
    }

    return {
        ensureDevice,
        createTransport,
        startSpeaker,
        ensureRecvTransport,
        consumeFrom
    }
}
