// client/composables/voice-sfu/audio.js

export function createAudioModule({audioUnlockedRef, logger = console}) {
    let audioCtx = null
    let bound = false

    const requestAudioUnlock = async () => {
        logger.log('[SFU] requestAudioUnlock called; current state:', {
            unlocked: audioUnlockedRef.value,
            ctxState: audioCtx?.state
        })
        if (audioUnlockedRef.value) return true

        const AC = window.AudioContext || window.webkitAudioContext
        if (!AC) {
            logger.warn('[SFU] AudioContext not supported, auto-unlock')
            audioUnlockedRef.value = true
            return true
        }

        if (!audioCtx) {
            audioCtx = new AC()
            logger.log('[SFU] AudioContext created', {state: audioCtx.state})
        }

        try {
            if (audioCtx.state === 'suspended') {
                await audioCtx.resume()
                logger.log('[SFU] AudioContext resumed')
            }
            const src = audioCtx.createBufferSource()
            src.buffer = audioCtx.createBuffer(1, 1, 22050)
            src.connect(audioCtx.destination)
            src.start(0)
            audioUnlockedRef.value = true
            logger.log('[SFU] audio unlocked OK')
            return true
        } catch (e) {
            logger.warn('[SFU] Audio unlock failed:', e)
            return false
        }
    }

    const getCtx = () => audioCtx

    // جسچر-آنلاک: روی document یا هر دکمه‌ای بایند کن
    const bindGestureUnlock = (el = document) => {
        if (bound) return
        const handler = async () => {
            try {
                await requestAudioUnlock()
            } finally {
                el.removeEventListener('pointerdown', handler)
                el.removeEventListener('touchstart', handler)
                el.removeEventListener('keydown', handler)
                bound = true
            }
        }
        el.addEventListener('pointerdown', handler, {passive: true})
        el.addEventListener('touchstart', handler, {passive: true})
        el.addEventListener('keydown', handler, {passive: true})
    }

    // به‌صورت امن یک المان صوتی را پخش کن (بعد از آنلاک)
    const safePlay = async (audioEl) => {
        if (!audioEl) return
        try {
            if (!audioUnlockedRef.value) await requestAudioUnlock()
            const p = audioEl.play()
            if (p && typeof p.then === 'function') await p
            logger.log('[SFU] audio.play() ok')
        } catch (err) {
            logger.warn('[SFU] audio.play() blocked, waiting for user gesture', err)
            // اگر بلاک شد، دوباره با جسچر سعی کن
            bindGestureUnlock()
        }
    }


    return {requestAudioUnlock, bindGestureUnlock, safePlay, getCtx}

}
